<?php
/**
 * Counter System for Check IGN API
 * Tracks page views and API requests
 */

// Counter file paths
define('PAGE_VIEWS_FILE', dirname(__FILE__) . '/counters/page_views.txt');
define('API_REQUESTS_FILE', dirname(__FILE__) . '/counters/api_requests.txt');

// Create counters directory if it doesn't exist
if (!is_dir(__DIR__ . '/counters')) {
    mkdir(__DIR__ . '/counters', 0755, true);
}

// Initialize counter files if they don't exist
if (!file_exists(PAGE_VIEWS_FILE)) {
    file_put_contents(PAGE_VIEWS_FILE, '0');
}
if (!file_exists(API_REQUESTS_FILE)) {
    file_put_contents(API_REQUESTS_FILE, '0');
}

/**
 * Get current counter value
 */
function getCounter($file) {
    if (file_exists($file)) {
        return (int)file_get_contents($file);
    }
    return 0;
}

/**
 * Increment counter
 */
function incrementCounter($file) {
    $current = getCounter($file);
    $new = $current + 1;

    // Try to write the file
    $result = file_put_contents($file, $new);

    if ($result === false) {
        error_log('Failed to write counter file: ' . $file);
        return false;
    }

    return $new;
}

/**
 * Increment API request counter (for internal use)
 */
function incrementApiRequest() {
    return incrementCounter(API_REQUESTS_FILE);
}

/**
 * Increment page view counter (for internal use)
 */
function incrementPageView() {
    return incrementCounter(PAGE_VIEWS_FILE);
}

/**
 * Get all counters as JSON
 */
function getAllCounters() {
    return [
        'page_views' => getCounter(PAGE_VIEWS_FILE),
        'api_requests' => getCounter(API_REQUESTS_FILE)
    ];
}

// Handle different actions
$action = $_GET['action'] ?? '';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST');
header('Access-Control-Allow-Headers: Content-Type');

switch ($action) {
    case 'increment_page_view':
        $result = incrementCounter(PAGE_VIEWS_FILE);
        echo json_encode(['success' => true, 'new_value' => $result]);
        break;

    case 'increment_api_request':
        $result = incrementCounter(API_REQUESTS_FILE);
        echo json_encode(['success' => true, 'new_value' => $result]);
        break;

    case 'get_counters':
    default:
        echo json_encode(getAllCounters());
        break;
}
?>