<?php

/**
 * Game Nickname Lookup API
 * Production Version
 * Based on proven debug code
 */

require 'mainconfig.php';
header('Content-Type: application/json');

// Enable error reporting but don't display
error_reporting(E_ALL);
ini_set('display_errors', 0);

// Start output buffering to catch any unexpected output
ob_start();

try {
    // Validate required POST parameters
    if (!isset($_POST['game'], $_POST['user_id'], $_POST['other_id'])) {
        throw new Exception('Missing required parameters: game, user_id, other_id');
    }
    
    // Increment API request counter
    $baseDir = dirname(__FILE__);
    $counterDir = $baseDir . '/counters';
    $counterFile = $counterDir . '/api_requests.txt';

    // Ensure counters directory exists
    if (!is_dir($counterDir)) {
        if (!mkdir($counterDir, 0755, true)) {
            error_log('Failed to create counters directory: ' . $counterDir);
            // Continue without counter
            goto skip_counter;
        }
    }

    // Ensure counter file exists
    if (!file_exists($counterFile)) {
        if (file_put_contents($counterFile, '0') === false) {
            error_log('Failed to create counter file: ' . $counterFile);
            goto skip_counter;
        }
    }

    // Increment counter
    $current = (int)file_get_contents($counterFile);
    $new = $current + 1;
    if (file_put_contents($counterFile, $new) !== false) {
        error_log('API Counter successfully incremented to: ' . $new);
    } else {
        error_log('Failed to write API counter file: ' . $counterFile);
    }

    skip_counter:
    
    // Sanitize input
    $input_post = array(
        'game' => trim($_POST['game']),
        'user_id' => trim($_POST['user_id']),
        'other_id' => trim($_POST['other_id'])
    );
    
    // Verify database connection
    if (!$db) {
        throw new Exception('Database connection failed');
    }
    
    // Check if account already exists in cache
    $checkAccount = $db->prepare("SELECT * FROM accounts WHERE code = ? AND user_id = ? AND other_id = ?");
    if (!$checkAccount) {
        throw new Exception('Database prepare error: ' . $db->error);
    }
    
    $checkAccount->bind_param("sss", $input_post['game'], $input_post['user_id'], $input_post['other_id']);
    
    if (!$checkAccount->execute()) {
        throw new Exception('Database query error: ' . $checkAccount->error);
    }
    
    $result_query = $checkAccount->get_result();
    $is_check = false;
    $is_new = false;
    $dataAccount = null;
    
    // Check if cached result exists and is not expired
    if ($result_query->num_rows == 1) {
        $dataAccount = $result_query->fetch_assoc();
        if ($dataAccount['expired_at'] > time()) {
            // Return cached result
            $result = array(
                'response' => true, 
                'data' => array(
                    'status' => 200, 
                    'nickname' => $dataAccount['nickname'],
                    'cached' => true
                )
            );
        } else {
            // Cache expired, need to check API
            $is_check = true;
        }
    } else {
        // New account, need to check API
        $is_check = true;
        $is_new = true;
    }
    
    // If we need to check the API
    if ($is_check) {
        // Load the API library
        require 'src/ApiGames.php';
        
        // Verify the class exists
        if (!class_exists('Aditdev\ApiGames')) {
            throw new Exception('ApiGames class not found in src/ApiGames.php');
        }
        
        // Create API instance
        $apiGames = new Aditdev\ApiGames;
        
        // Verify the game method exists
        if (!method_exists($apiGames, $input_post['game'])) {
            throw new Exception('Game not supported: ' . $input_post['game']);
        }
        
        // Call the API method
        try {
            $response = $apiGames->{$input_post['game']}($input_post['user_id'], $input_post['other_id']);
        } catch (Exception $e) {
            throw new Exception('API call failed: ' . $e->getMessage());
        }
        
        // Parse the JSON response
        $result = json_decode($response, true);
        
        if ($result === null) {
            throw new Exception('Invalid JSON from API: ' . substr($response, 0, 100));
        }
        
        // Check if API returned success
        if ($result['status'] == 200) {
            // Save to database
            if ($is_new) {
                // Insert new account
                $insertStmt = $db->prepare("INSERT INTO accounts (code, user_id, other_id, nickname, expired_at) VALUES (?, ?, ?, ?, ?)");
                if (!$insertStmt) {
                    throw new Exception('Database insert prepare error: ' . $db->error);
                }
                
                $expiry = strtotime('+7 day');
                $insertStmt->bind_param("ssssi", $input_post['game'], $input_post['user_id'], $input_post['other_id'], $result['nickname'], $expiry);
                
                if (!$insertStmt->execute()) {
                    throw new Exception('Database insert error: ' . $insertStmt->error);
                }
                
                $insertStmt->close();
            } else {
                // Update existing account
                $updateStmt = $db->prepare("UPDATE accounts SET nickname = ?, expired_at = ? WHERE id = ?");
                if (!$updateStmt) {
                    throw new Exception('Database update prepare error: ' . $db->error);
                }
                
                $expiry = strtotime('+7 day');
                $updateStmt->bind_param("sii", $result['nickname'], $expiry, $dataAccount['id']);
                
                if (!$updateStmt->execute()) {
                    throw new Exception('Database update error: ' . $updateStmt->error);
                }
                
                $updateStmt->close();
            }
            
            // Return success with API data
            $result = array('response' => true, 'data' => $result);
        } else {
            // API returned an error status
            $result = array('response' => false, 'data' => $result);
        }
    }
    
    // Clear any buffered output
    ob_end_clean();
    
    // Return the result
    exit(json_encode($result, JSON_PRETTY_PRINT));
    
} catch (Exception $e) {
    // Clear buffered output
    ob_end_clean();
    
    // Log the error (for debugging)
    error_log('API Error: ' . $e->getMessage());
    
    // Return error response
    http_response_code(500);
    exit(json_encode(array(
        'response' => false,
        'data' => array(
            'status' => 500,
            'msg' => $e->getMessage()
        )
    ), JSON_PRETTY_PRINT));
}

?>
